'use strict';

/* --------------------------------------------------------------
 collapse_left_menu.js 2015-10-15 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Collapse Main Left Menu
 *
 * This module will handle the collapse and expansion of the main left menu of the admin section. The HTML
 * for the collapse button comes from the "html/compatibility/collapse_left_menu.php".
 *
 * @module Compatibility/collapse_left_menu
 */
gx.compatibility.module('collapse_left_menu', ['user_configuration_service'],

/**  @lends module:Compatibility/collapse_left_menu */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // ELEMENTS DEFINITION
    // ------------------------------------------------------------------------

    var $this = $(this),
        $menu = $('.main-left-menu'),
        $currentMenuBox = $menu.find('.leftmenu_box.current'),
        $menuToggleButton = $this.find('.menu-toggle-button'),
        $menuButtonIndicator = $menuToggleButton.find('#menu-button-indicator'),
        menuInitState = $menu.data('initState');

    // ------------------------------------------------------------------------
    // VARIABLES DEFINITION
    // ------------------------------------------------------------------------

    var module = {},
        initialCssWidth = $menu.css('width'),
        userConfigurationService = jse.libs.user_configuration_service,
        userConfig = {
        userId: data.userId,
        configurationKey: 'menuVisibility'
    },
        stateMap = {
        collapse: {
            next: 'expand',
            button: 'right',
            class: 'collapsed',
            do: function _do(isAnimated) {
                _collapse(isAnimated);
            }
        },
        expand: {
            next: 'expandAll',
            button: 'down',
            class: 'expanded',
            do: function _do(isAnimated) {
                _expand(isAnimated);
            }
        },
        expandAll: {
            next: 'collapse',
            button: 'left',
            class: 'expanded-all',
            do: function _do(isAnimated) {
                _expandAll(isAnimated);
            }
        }
    },
        currentState;

    // ------------------------------------------------------------------------
    // HELPERS
    // ------------------------------------------------------------------------

    var isMenuVisible = function isMenuVisible() {
        return !$menu.hasClass('collapsed');
    };

    // ------------------------------------------------------------------------
    // STATE CHANGE TRIGGERS
    // ------------------------------------------------------------------------

    var _changeState = function _changeState(state, isAnimated) {
        var saveConfig = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;

        currentState = state;
        stateMap[currentState].do(isAnimated);

        if (saveConfig) {
            _saveConfig();
        }

        _changeButton();
    };

    var _changeButton = function _changeButton() {
        var className = 'fa fa-caret-';
        var arrowDirection = stateMap[currentState].button;
        $menuButtonIndicator.removeAttr('class').addClass(className + arrowDirection);
    };

    // ------------------------------------------------------------------------
    // COLLAPSE / EXPAND MENU
    // ------------------------------------------------------------------------

    /**
     * Collapse Left Menu
     * @param {boolean} isAnimated - Animate the hiding?
     * @private
     */
    var _collapse = function _collapse(isAnimated) {

        var currentBox = $this.parent().find('ul.current');

        // Collapse menu
        if (isAnimated) {
            $menu.animate({
                'width': '45px'
            }, 300, 'swing');
        } else {
            $menu.css('width', '45px');
            $('.columnLeft2').css('width', '45px');
        }
        currentBox.hide();

        $(document).trigger('leftmenu:collapse');

        // Fade out heading text
        $menu.find('.leftmenu_head span').fadeOut('fast');

        // Class changes
        $menu.removeClass('expanded-all').addClass('collapsed');

        $menu.find('.current:not(li)').removeClass('current');

        $menu.find('.current-menu-head').addClass('current');

        var interval = setInterval(function () {
            if (currentState === 'collapse') {
                if ($('.leftmenu_head.current').length > 1) {
                    $menu.find('.leftmenu_head.current:not(.current-menu-head)').removeClass('current');
                    clearInterval(interval);
                }
            } else {
                clearInterval(interval);
            }
        }, 1);
    };

    /**
     * Expand Left Menu
     * @private
     */
    var _expand = function _expand() {

        var currentBox = $this.parent().find('ul.current');

        // Expand menu
        $menu.animate({
            'width': initialCssWidth
        }, 300, 'swing');
        currentBox.show();

        // Fade in heading text
        $menu.find('.leftmenu_head span').fadeIn('slow');

        $(document).trigger('leftmenu:expand');

        // Class changes
        $menu.removeClass('collapsed');
        $currentMenuBox.addClass('current');
    };

    /**
     * Expand all menu items
     * @private
     */
    var _expandAll = function _expandAll(isAnimated) {

        $menu.addClass('expanded-all');

        var $headingBoxes = $menu.find('div.leftmenu_head:not(.current)');

        if (isAnimated) {
            $headingBoxes.addClass('current', 750, 'swing');
        } else {
            $headingBoxes.addClass('current');
        }

        $(document).trigger('leftmenu:expand');

        $menu.find('ul.leftmenu_box:not(.current)').addClass('current');
    };

    // ------------------------------------------------------------------------
    // USER CONFIGURATION HANDLER
    // ------------------------------------------------------------------------

    var _saveConfig = function _saveConfig() {
        userConfigurationService.set({
            data: $.extend(userConfig, {
                configurationValue: currentState
            })
        });
    };

    // ------------------------------------------------------------------------
    // EVENT HANDLERS
    // ------------------------------------------------------------------------

    var _onClick = function _onClick(event) {
        if ($menuToggleButton.has(event.target).length) {
            _changeState(stateMap[currentState].next, true);
        }
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {

        $('div.leftmenu_head.current').addClass('current-menu-head');

        if (!isMenuVisible()) {
            $currentMenuBox.removeClass('current');
        }

        currentState = menuInitState;

        if (currentState === '') {
            currentState = 'expand'; // Default value if there is no menuInitState set yet.
        }

        _changeState(currentState, false, false);

        $this.on('click', _onClick);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
